#include "mCoreLiteIR.h"

static mCoreLiteIR* ir_inst = nullptr;

ISR(TIMER2_OVF_vect)
{
    if (ir_inst) ir_inst->handleTick();
}

mCoreLiteIR::mCoreLiteIR(uint8_t pin)
{
    _pin = pin;
    ir_inst = this;
}

void mCoreLiteIR::begin()
{
    pinMode(_pin, INPUT);

    cli();
    TCCR2A = 0;
    TCCR2B = (1 << CS21);   // prescaler 8 → overflow ≈ 128 µs
    TIMSK2 = (1 << TOIE2);  // IRQ enable
    sei();

    rawlen = 0;
    state = 0;
}

void mCoreLiteIR::handleTick()
{
    static uint8_t tick = 0;

    uint8_t irdata = digitalRead(_pin);
    tick++;

    if (tick > 250) tick = 250;
    if (rawlen >= 200) { state = 3; return; }

    switch (state)
    {
        case 0: // IDLE waiting MARK
            if (irdata == LOW) {
                rawlen = 0;
                tick = 0;
                state = 1;
            }
            break;

        case 1: // MARK
            if (irdata == HIGH) {
                rawbuf[rawlen++] = tick;
                tick = 0;
                state = 2;
            }
            break;

        case 2: // SPACE
            if (irdata == LOW) {
                rawbuf[rawlen++] = tick;
                tick = 0;
                state = 1;
            } 
            else if (tick > 180) {
                state = 3; // END FRAME
            }
            break;

        case 3:
            break;
    }
}

bool mCoreLiteIR::available()
{
    return (state == 3);
}

static bool match(uint16_t t, uint16_t target)
{
    return t > target - target/4 && t < target + target/4;
}

uint32_t mCoreLiteIR::readCode()
{
    uint32_t v = 0;
    if (decodeNEC(v)) {
        state = 0;
        rawlen = 0;
        return v;
    }
    state = 0;
    rawlen = 0;
    return 0;
}

bool mCoreLiteIR::decodeNEC(uint32_t& value)
{
    if (rawlen < 68) return false;

    int i = 0;

    // Header
    if (!match(rawbuf[i++], 180)) return false;  // 9ms
    if (!match(rawbuf[i++], 90))  return false;  // 4.5ms

    uint32_t data = 0;

    for (int b = 0; b < 32; b++)
    {
        // MARK 560µs
        if (!match(rawbuf[i++], 11)) return false;

        // SPACE 560 / 1690
        if (match(rawbuf[i], 34))     // bit = 1
            data |= (1UL << b);
        else if (!match(rawbuf[i], 11)) // bit = 0
            return false;

        i++;
    }

    value = data;
    return true;
}
char mCoreLiteIR::readChar()
{
    uint32_t c = readCode();
    if (c == 0) return 0;

    switch (c)
    {
        //case 0x00FF18E7: return '^';
        //case 0x00FF4AB5: return 'v';
        //case 0x00FF10EF: return '<';
        //case 0x00FF5AA5: return '>';

        case 0x00FFA25D: return 'A';
        case 0x00FF629D: return 'B';
        case 0x00FFE21D: return 'C';
        case 0x00FF22DD: return 'D';

        case 0x00FF30CF: return '1';
        case 0x00FF18E7: return '2';
        case 0x00FF7A85: return '3';
        case 0x00FF10EF: return '4';
        case 0x00FF38C7: return '5';
        case 0x00FF5AA5: return '6';
        case 0x00FF42BD: return '7';
        case 0x00FF4AB5: return '8';
        case 0x00FF52AD: return '9';
    }

    return '?';
}
