/**
 * mCoreLite - Librairie mBot (mCore)
 * Auteur : Sébastien Tack - novembre 2025 (sebastien.tack@free.fr)
 *
 * E/S utilisées :
 *  Moteurs    : D7/D6 (M1) et D4/D5 (M2)
 *  Ligne      : D9 / D10
 *  LDR        : A6
 *  Bouton     : A7
 *  Buzzer     : D8
 *  LEDs       : D13 (WS2812)
 *  IR         : D2 (réception) / D3 (émission)
 *  Ultrason   : A3 (mono-fil TRIG/ECHO combinés – mBot v3)
 *  Bluetooth / Serial USB : D0 (RX), D1 (TX)
 *
 * Pins totalement libres :
 *  D11, D12,
 *  A0, A1,A2,
 *  A4, A5 (I2C) A ne pas utiliser sauf stricte nécessité
 *  --------------------------------------------------------------------------
 *  Notes :
 *    - La bibliothèque IRremote est utilisée pour le décodage NEC.
 *    - Le capteur ultrason est géré en mode “monofil” compatible mBot.
 *    - Les LED WS2812 utilisent Adafruit_NeoPixel.
 *  --------------------------------------------------------------------------
 */
 
#include "mCoreLite.h"
#include <Adafruit_NeoPixel.h>
#include <IRremote.hpp>

Adafruit_NeoPixel strip = Adafruit_NeoPixel(2, LED_WS2812_PIN, NEO_GRB + NEO_KHZ800);

void mCoreLite::motors(int left, int right){
    digitalWrite(M1_DIR, left >= 0);
    analogWrite(M1_PWM, abs(left));
    digitalWrite(M2_DIR, right >= 0);
    analogWrite(M2_PWM, abs(right));
}

void mCoreLite::forward(int speed){
    motors(-speed,speed);
}
void mCoreLite::reverse(int speed){
    motors(speed,-speed);
}
void mCoreLite::left(int speed){
    motors(speed,speed);
}
void mCoreLite::right(int speed){
    motors(-speed,-speed);
}
void mCoreLite::stop(){
    motors(0,0);
}

void mCoreLite::motor(int id, int speed){
    if(id == 1) motors(speed,0);
    else motors(0,speed);
}

int mCoreLite::button(){
    return analogRead(BUTTON_PIN);  // LOW = pressé
}

int mCoreLite::lineLeft(){
    if (!useLineSensors) return -1;  // capteur désactivé
    return digitalRead(LINE_L);
}

int mCoreLite::lineRight(){
    if (!useLineSensors) return -1;
    return digitalRead(LINE_R);
}

int mCoreLite::light(){
    return analogRead(LDR_PIN);
}

void mCoreLite::toneBuz(int f,int d){
    tone(BUZZER_PIN,f,d);
}

void mCoreLite::beginLED(){
    strip.begin();
    strip.show();
}

void mCoreLite::setLED(int n,uint8_t r,uint8_t g,uint8_t b){
    strip.setPixelColor(n, strip.Color(r,g,b));
    strip.show();
}
long mCoreLite::distanceCM() {

    // Signal à 1 fil (TRIG + ECHO sur S2)
    pinMode(US_PIN, OUTPUT);

    digitalWrite(US_PIN, LOW);
    delayMicroseconds(2);

    digitalWrite(US_PIN, HIGH);
    delayMicroseconds(10);

    digitalWrite(US_PIN, LOW);
    delayMicroseconds(2);

    pinMode(US_PIN, INPUT);

    long duration = pulseIn(US_PIN, HIGH, 30000);

    if(duration == 0) return 400;

    long dist = duration / 58;
    if(dist > 400) dist = 400;

    return dist;
}

char mCoreLite::decodeMbot(uint8_t cmd)
{
    switch(cmd)
    {
        case 0x45: return 'A';
        case 0x46: return 'B';
        case 0x47: return 'C';
        case 0x44: return 'D';
        case 0x0D: return 'F';
        case 0x43: return 'E';

        case 0x40: return '^';
        case 0x19: return 'v';
        case 0x07: return '<';
        case 0x09: return '>';
        case 0x15: return '*';

        case 0x16: return '0';      
        case 0x0C: return '1';
        case 0x18: return '2';
        case 0x5E: return '3';
        case 0x08: return '4';
        case 0x1C: return '5';
        case 0x5A: return '6';
        case 0x42: return '7';
        case 0x52: return '8';
        case 0x4A: return '9';

        default: return '?';
    }
}


uint8_t mCoreLite::IrReadCode() 
{
    if (!IrReceiver.decode())
        return 0;

    if (IrReceiver.decodedIRData.flags & IRDATA_FLAGS_IS_REPEAT) {
        IrReceiver.resume();
        return 0;
    }

    uint8_t result = IrReceiver.decodedIRData.command;
    IrReceiver.resume();

    // --- Ajout du buffer interne ---
    car_send = decodeMbot(result);   // conversion en caractère
    flag = true;                    // événement reçu

    Serial.print("CMD = 0x");
    Serial.println(result, HEX);
    Serial.println(car_send);

    return result;
}

void mCoreLite::botSetupIr() {
	IrReceiver.begin(IR_RECEIVE_PIN);
}

void mCoreLite::botSetup() {

    // Moteurs
    pinMode(M1_DIR, OUTPUT);
    pinMode(M1_PWM, OUTPUT);
    pinMode(M2_DIR, OUTPUT);
    pinMode(M2_PWM, OUTPUT);

    // Capteurs de ligne
    pinMode(LINE_L, INPUT);
    pinMode(LINE_R, INPUT);
	
	  
    // Buzzer
    pinMode(BUZZER_PIN, OUTPUT);

    // Bouton
    //pinMode(BUTTON_PIN, INPUT);

    // LDR
    pinMode(LDR_PIN, INPUT);
    beginLED();
	Serial.begin(9600);
	//IrReceiver.begin(IR_RECEIVE_PIN);
}

void mCoreLite::enableLineSensors(bool enable)
{
    useLineSensors = enable;

    if (enable) {
        pinMode(LINE_L, INPUT);
        pinMode(LINE_R, INPUT);
    } else {
        // Libère complètement D9 et D10
        pinMode(LINE_L, INPUT_PULLUP);  // ou OUTPUT selon ton servo driver
        pinMode(LINE_R, INPUT_PULLUP);
    }
}


void mCoreLite::pause(uint32_t ms)
{
    uint32_t t0 = millis();

    while (millis() - t0 < ms) {
        // Permet de garder un robot réactif pendant la pause
        IrReadCode();   // l’IR continue à fonctionner
        // Plus tard on peut rajouter ici un scheduler

        yield(); // Laisse respirer Arduino (optionnel)
    }
}

