/*
 * Decompiled with CFR 0.152.
 */
package de.neemann.digital.core;

import de.neemann.digital.core.Bits;
import de.neemann.digital.core.Value;
import de.neemann.digital.core.ValueFormatter;
import de.neemann.digital.core.element.ElementAttributes;
import de.neemann.digital.core.element.Keys;

public enum IntFormat {
    def(ValueFormatterDefault.access$000()),
    dec(new ValueFormatterDecimal(false)),
    decSigned(new ValueFormatterDecimal(true)),
    hex(ValueFormatterHex.access$200()),
    bin(new ValueFormatterBinary()),
    oct(new ValueFormatterOctal()),
    ascii(new ValueFormatterAscii()),
    fixed(attributes -> new ValueFormatterFixedPoint(attributes, false)),
    fixedSigned(attributes -> new ValueFormatterFixedPoint(attributes, true)),
    floating(new ValueFormatterFloat());

    public static final ValueFormatter DEFAULT_FORMATTER;
    public static final ValueFormatter HEX_FORMATTER;
    private final Factory factory;
    private final boolean dependsOnAttributes;
    private static final char[] DIGITS;
    private static final int BUF = 16;

    private IntFormat(ValueFormatter instance) {
        this.factory = attributes -> instance;
        this.dependsOnAttributes = false;
    }

    private IntFormat(Factory factory) {
        this.factory = factory;
        this.dependsOnAttributes = true;
    }

    public ValueFormatter createFormatter(ElementAttributes attributes) {
        return this.factory.create(attributes);
    }

    public boolean dependsOnAttributes() {
        return this.dependsOnAttributes;
    }

    private static long dragValueSigned(long initial, int bits, double inc, boolean signed) {
        long min;
        long max;
        if (signed) {
            long mask = Bits.mask(bits);
            long signedFlag = Bits.signedFlagMask(bits);
            if ((initial & signedFlag) != 0L) {
                initial |= mask ^ 0xFFFFFFFFFFFFFFFFL;
            }
            max = mask >>> 1;
            min = -max - 1L;
        } else {
            max = Bits.mask(bits);
            min = 0L;
        }
        return Math.max(min, Math.min(max, initial + Math.round((double)max * inc)));
    }

    public static String toShortHex(long value) {
        return IntFormat.toShortHex(value, false);
    }

    private static String toShortHex(long value, boolean omitPrefix) {
        if (value == 0L) {
            return "0";
        }
        boolean wasChar = false;
        int p = 16;
        char[] data = new char[16];
        while (value != 0L) {
            int d = (int)(value & 0xFL);
            if (d >= 10) {
                wasChar = true;
            }
            data[--p] = DIGITS[d];
            value >>>= 4;
        }
        if (omitPrefix || wasChar || p == 15) {
            return new String(data, p, 16 - p);
        }
        return "0x" + new String(data, p, 16 - p);
    }

    private static int decStrLen(int bits) {
        if (bits == 64) {
            return 20;
        }
        if (bits == 63) {
            return 19;
        }
        return (int)Math.ceil(Math.log10(1L << bits));
    }

    static {
        DEFAULT_FORMATTER = ValueFormatterDefault.INSTANCE;
        HEX_FORMATTER = ValueFormatterHex.INSTANCE;
        DIGITS = new char[]{'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'};
    }

    private static final class ValueFormatterFloat
    implements ValueFormatter {
        private static final int SIZE32 = Float.toString((float)(-Math.PI)).length();
        private static final int SIZE64 = Double.toString(-Math.PI).length();

        private ValueFormatterFloat() {
        }

        @Override
        public String formatToView(Value inValue) {
            if (inValue.isHighZ()) {
                return inValue.toString();
            }
            switch (inValue.getBits()) {
                case 32: {
                    return Float.toString(Float.intBitsToFloat((int)inValue.getValue()));
                }
                case 64: {
                    return Double.toString(Double.longBitsToDouble(inValue.getValue()));
                }
            }
            return HEX_FORMATTER.formatToView(inValue);
        }

        @Override
        public String formatToEdit(Value inValue) {
            if (inValue.isHighZ()) {
                return "Z";
            }
            switch (inValue.getBits()) {
                case 32: {
                    float f = Float.intBitsToFloat((int)inValue.getValue());
                    if (Float.isFinite(f)) {
                        return Float.toString(f);
                    }
                    return HEX_FORMATTER.formatToEdit(inValue);
                }
                case 64: {
                    double d = Double.longBitsToDouble(inValue.getValue());
                    if (Double.isFinite(d)) {
                        return d + "d";
                    }
                    return HEX_FORMATTER.formatToEdit(inValue);
                }
            }
            return HEX_FORMATTER.formatToEdit(inValue);
        }

        @Override
        public int strLen(int bits) {
            switch (bits) {
                case 32: {
                    return SIZE32;
                }
                case 64: {
                    return SIZE64;
                }
            }
            return HEX_FORMATTER.strLen(bits);
        }

        @Override
        public boolean isSuitedForAddresses() {
            return false;
        }

        @Override
        public long dragValue(long initialInt, int bits, double inc) {
            double initial;
            if (bits == 32) {
                initial = Float.intBitsToFloat((int)initialInt);
            } else if (bits == 64) {
                initial = Double.longBitsToDouble(initialInt);
            } else {
                return HEX_FORMATTER.dragValue(initialInt, bits, inc);
            }
            if (!Double.isFinite(initial)) {
                initial = 0.0;
            }
            double fac = Math.exp(Math.abs(inc) * 15.0) / 1000.0;
            double delta = Math.abs(initial == 0.0 ? 1.0 : initial) * fac * Math.signum(inc);
            double exp = Math.pow(10.0, Math.floor(Math.log10(Math.abs(delta))));
            double val = (double)Math.round((initial + delta) / exp) * exp;
            if (bits == 32) {
                return Float.floatToIntBits((float)val);
            }
            return Double.doubleToLongBits(val);
        }

        @Override
        public boolean isSeparatorInFrontOf(int bits, int bit) {
            switch (bits) {
                case 32: {
                    return bit == 31 || bit == 23;
                }
                case 64: {
                    return bit == 63 || bit == 52;
                }
            }
            return HEX_FORMATTER.isSeparatorInFrontOf(bits, bit);
        }
    }

    private static final class ValueFormatterFixedPoint
    implements ValueFormatter {
        private static final int[] TABLE = new int[]{0, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 14, 15, 16, 17, 17, 18, 19, 19, 20, 20, 20, 20, 20, 20, 20, 20, 20, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 22};
        private final int fixedPoint;
        private final boolean signed;
        private final double divisor;

        private ValueFormatterFixedPoint(ElementAttributes attr, boolean signed) {
            this.fixedPoint = attr.get(Keys.FIXED_POINT);
            this.divisor = Bits.up(1L, this.fixedPoint);
            this.signed = signed;
        }

        @Override
        public String formatToView(Value inValue) {
            if (inValue.isHighZ()) {
                return inValue.toString();
            }
            return this.format(inValue);
        }

        @Override
        public String formatToEdit(Value inValue) {
            if (inValue.isHighZ()) {
                return "Z";
            }
            return this.format(inValue) + ":" + this.fixedPoint;
        }

        @Override
        public int strLen(int bits) {
            int fp = this.fixedPoint;
            if (fp >= TABLE.length) {
                fp = TABLE.length - 1;
            }
            return IntFormat.decStrLen(Math.max(1, bits - fp)) + TABLE[fp];
        }

        @Override
        public boolean isSuitedForAddresses() {
            return false;
        }

        private String format(Value inValue) {
            if (this.signed) {
                return Double.toString((double)inValue.getValueSigned() / this.divisor);
            }
            return Double.toString((double)inValue.getValue() / this.divisor);
        }

        @Override
        public long dragValue(long initial, int bits, double inc) {
            return IntFormat.dragValueSigned(initial, bits, inc, this.signed);
        }

        @Override
        public boolean isSeparatorInFrontOf(int bits, int bit) {
            return bit == this.fixedPoint;
        }
    }

    private static final class ValueFormatterDecimal
    extends ValueFormatterViewEdit {
        private final boolean signed;

        private ValueFormatterDecimal(boolean signed) {
            super(true);
            this.signed = signed;
        }

        @Override
        public int strLen(int bits) {
            if (this.signed) {
                return IntFormat.decStrLen(bits - 1) + 1;
            }
            return IntFormat.decStrLen(bits);
        }

        @Override
        protected String format(Value value) {
            if (this.signed) {
                return Long.toString(value.getValueSigned());
            }
            return Long.toString(value.getValue());
        }

        @Override
        public long dragValue(long initial, int bits, double inc) {
            return IntFormat.dragValueSigned(initial, bits, inc, this.signed);
        }
    }

    private static final class ValueFormatterAscii
    extends ValueFormatterViewEdit {
        private ValueFormatterAscii() {
            super(false);
        }

        @Override
        public int strLen(int bits) {
            return 3;
        }

        @Override
        protected String format(Value value) {
            return "'" + (char)value.getValue() + "'";
        }
    }

    private static final class ValueFormatterBinary
    extends ValueFormatterViewEdit {
        private ValueFormatterBinary() {
            super(false);
        }

        @Override
        public int strLen(int bits) {
            return bits + 2;
        }

        @Override
        protected String format(Value inValue) {
            int bits = inValue.getBits();
            char[] data = new char[bits];
            long value = inValue.getValue();
            long mask = 1L;
            for (int i = bits - 1; i >= 0; --i) {
                data[i] = (value & mask) != 0L ? 49 : 48;
                mask <<= 1;
            }
            return "0b" + new String(data);
        }
    }

    private static final class ValueFormatterOctal
    extends ValueFormatterViewEdit {
        private ValueFormatterOctal() {
            super(true);
        }

        @Override
        public int strLen(int bits) {
            return (bits - 1) / 3 + 3;
        }

        @Override
        protected String format(Value inValue) {
            int bits = inValue.getBits();
            int numChars = (bits - 1) / 3 + 1;
            StringBuilder sb = new StringBuilder("0");
            long value = inValue.getValue();
            for (int i = numChars - 1; i >= 0; --i) {
                int c = (int)(value >> i * 3 & 7L);
                sb.append(DIGITS[c]);
            }
            return sb.toString();
        }

        @Override
        public boolean isSeparatorInFrontOf(int bits, int bit) {
            return bit % 3 == 0;
        }
    }

    private static final class ValueFormatterHex
    extends ValueFormatterViewEdit {
        private static final ValueFormatterHex INSTANCE = new ValueFormatterHex();

        private ValueFormatterHex() {
            super(true);
        }

        @Override
        protected String format(Value inValue) {
            int bits = inValue.getBits();
            int numChars = (bits - 1) / 4 + 1;
            StringBuilder sb = new StringBuilder("0x");
            long value = inValue.getValue();
            for (int i = numChars - 1; i >= 0; --i) {
                int c = (int)(value >> i * 4 & 0xFL);
                sb.append(DIGITS[c]);
            }
            return sb.toString();
        }

        @Override
        public int strLen(int bits) {
            return (bits - 1) / 4 + 3;
        }

        @Override
        public boolean isSeparatorInFrontOf(int bits, int bit) {
            return bit % 4 == 0;
        }
    }

    private static abstract class ValueFormatterViewEdit
    implements ValueFormatter {
        private final boolean suitedForAddresses;

        private ValueFormatterViewEdit(boolean suitedForAddresses) {
            this.suitedForAddresses = suitedForAddresses;
        }

        @Override
        public String formatToView(Value inValue) {
            if (inValue.isHighZ()) {
                return inValue.toString();
            }
            return this.format(inValue);
        }

        @Override
        public String formatToEdit(Value inValue) {
            if (inValue.isHighZ()) {
                return "Z";
            }
            return this.format(inValue);
        }

        @Override
        public boolean isSuitedForAddresses() {
            return this.suitedForAddresses;
        }

        protected abstract String format(Value var1);

        @Override
        public long dragValue(long initial, int bits, double inc) {
            return IntFormat.dragValueSigned(initial, bits, inc, false);
        }
    }

    private static final class ValueFormatterDefault
    implements ValueFormatter {
        private static final ValueFormatter INSTANCE = new ValueFormatterDefault();

        private ValueFormatterDefault() {
        }

        @Override
        public String formatToView(Value inValue) {
            if (inValue.isHighZ()) {
                return inValue.toString();
            }
            return IntFormat.toShortHex(inValue.getValue(), false);
        }

        @Override
        public String formatToEdit(Value inValue) {
            if (inValue.isHighZ()) {
                return "Z";
            }
            long value = inValue.getValue();
            if (value >= 0L && value < 10L) {
                return Long.toString(value);
            }
            return "0x" + IntFormat.toShortHex(value, true);
        }

        @Override
        public int strLen(int bits) {
            return (bits - 1) / 4 + 3;
        }

        @Override
        public boolean isSuitedForAddresses() {
            return false;
        }

        @Override
        public long dragValue(long initial, int bits, double inc) {
            return IntFormat.dragValueSigned(initial, bits, inc, false);
        }

        @Override
        public boolean isSeparatorInFrontOf(int bits, int bit) {
            return bit % 4 == 0;
        }
    }

    private static interface Factory {
        public ValueFormatter create(ElementAttributes var1);
    }
}

