/*
 * Decompiled with CFR 0.152.
 */
package de.neemann.digital.analyse;

import de.neemann.digital.analyse.AnalyseException;
import de.neemann.digital.analyse.BoolTableExpanded;
import de.neemann.digital.analyse.CycleDetector;
import de.neemann.digital.analyse.DependencyAnalyser;
import de.neemann.digital.analyse.LabelNumbering;
import de.neemann.digital.analyse.ModelAnalyserInfo;
import de.neemann.digital.analyse.PathLenAnalyser;
import de.neemann.digital.analyse.SplitPinString;
import de.neemann.digital.analyse.TruthTable;
import de.neemann.digital.analyse.expression.BitSetter;
import de.neemann.digital.analyse.quinemc.BoolTableByteArray;
import de.neemann.digital.core.BacktrackException;
import de.neemann.digital.core.Model;
import de.neemann.digital.core.Node;
import de.neemann.digital.core.NodeException;
import de.neemann.digital.core.NodeWithoutDelay;
import de.neemann.digital.core.ObservableValue;
import de.neemann.digital.core.ObservableValues;
import de.neemann.digital.core.Signal;
import de.neemann.digital.core.flipflops.FlipflopD;
import de.neemann.digital.core.switching.NFET;
import de.neemann.digital.core.switching.Relay;
import de.neemann.digital.core.switching.RelayDT;
import de.neemann.digital.core.wiring.Clock;
import de.neemann.digital.core.wiring.Splitter;
import de.neemann.digital.draw.elements.PinException;
import de.neemann.digital.gui.Main;
import de.neemann.digital.lang.Lang;
import java.util.ArrayList;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ModelAnalyser {
    private static final Logger LOGGER = LoggerFactory.getLogger(ModelAnalyser.class);
    private static final int MAX_INPUTS_ALLOWED = 24;
    private final Model model;
    private final ArrayList<Signal> inputs;
    private final ArrayList<Signal> outputs;
    private ModelAnalyserInfo modelAnalyzerInfo;

    public ModelAnalyser(Model model) throws AnalyseException {
        this.model = model;
        this.modelAnalyzerInfo = new ModelAnalyserInfo(model);
        this.inputs = this.checkBinaryInputs(model.getInputs());
        this.checkUnique(this.inputs);
        this.outputs = this.checkBinaryOutputs(model.getOutputs());
        this.modelAnalyzerInfo.setInOut(this.inputs, this.outputs);
        for (Node n : model) {
            if (!n.hasState() || n.getClass() == FlipflopD.class) continue;
            throw new AnalyseException(Lang.get("err_cannotAnalyse_N", n.getClass().getSimpleName()));
        }
        int i = 0;
        List<FlipflopD> flipflops = model.findNode(FlipflopD.class);
        flipflops = this.replaceMultiBitFlipflops(flipflops);
        for (FlipflopD ff : flipflops) {
            this.checkClock(ff);
            if (ff.getDataBits() != 1) {
                throw new AnalyseException(Lang.get("err_MultiBitFlipFlopFound", new Object[0]));
            }
            ff.getDInput().removeObserver(ff);
            String label = this.getUniqueNameFor(ff);
            this.outputs.add(i++, new Signal(ModelAnalyser.addOne(label), ff.getDInput()));
            this.modelAnalyzerInfo.setSequentialInitValue(label, ff.getDefault());
            final ObservableValue q = (ObservableValue)ff.getOutputs().get(0);
            Signal sig = new Signal(label, q);
            if (this.inputs.contains(sig)) {
                throw new AnalyseException(Lang.get("err_varName_N_UsedTwice", sig.getName()));
            }
            this.inputs.add(sig);
            final ObservableValue notQ = (ObservableValue)ff.getOutputs().get(1);
            q.addObserver(new NodeWithoutDelay(new ObservableValue[]{notQ}){

                @Override
                public void hasChanged() {
                    notQ.setValue(q.getValue() ^ 0xFFFFFFFFFFFFFFFFL);
                }
            });
        }
        if (this.inputs.size() == 0) {
            throw new AnalyseException(Lang.get("err_analyseNoInputs", new Object[0]));
        }
        if (this.outputs.size() == 0) {
            throw new AnalyseException(Lang.get("err_analyseNoOutputs", new Object[0]));
        }
    }

    public static String addOne(String name) {
        if (name.endsWith("^n")) {
            return name.substring(0, name.length() - 1) + "{n+1}";
        }
        return name + "+1";
    }

    private String getUniqueNameFor(FlipflopD ff) {
        String label = ff.getLabel();
        if (label.length() == 0) {
            label = this.createOutputBasedName(ff);
        }
        if (!label.endsWith("n")) {
            label = label + "^n";
        }
        return new LabelNumbering(label).create(this::inputExist);
    }

    private boolean inputExist(String label) {
        for (Signal i : this.inputs) {
            if (!i.getName().equals(label)) continue;
            return true;
        }
        return false;
    }

    private String createOutputBasedName(FlipflopD ff) {
        ObservableValue q = (ObservableValue)ff.getOutputs().get(0);
        for (Signal o : this.outputs) {
            if (o.getValue() != q) continue;
            return o.getName();
        }
        return "Z";
    }

    private void checkUnique(ArrayList<Signal> signals) throws AnalyseException {
        for (int i = 0; i < signals.size() - 1; ++i) {
            for (int j = i + 1; j < signals.size(); ++j) {
                if (!signals.get(i).equals(signals.get(j))) continue;
                throw new AnalyseException(Lang.get("err_varName_N_UsedTwice", signals.get(i).getName()));
            }
        }
    }

    private ArrayList<Signal> checkBinaryOutputs(ArrayList<Signal> list) throws AnalyseException {
        ArrayList<Signal> outputs = new ArrayList<Signal>();
        for (Signal s : list) {
            int bits = s.getValue().getBits();
            if (bits == 1) {
                outputs.add(s);
                continue;
            }
            try {
                Splitter sp = Splitter.createOneToN(bits);
                sp.setInputs(s.getValue().asList());
                SplitPinString pins = SplitPinString.create(s);
                ObservableValues spOutputs = sp.getOutputs();
                String name = s.getName();
                if (!name.contains("_")) {
                    name = name + "_";
                }
                for (int i = spOutputs.size() - 1; i >= 0; --i) {
                    outputs.add(new Signal(name + i, (ObservableValue)spOutputs.get(i)).setPinNumber(pins.getPin(i)));
                }
                s.getValue().fireHasChanged();
                ArrayList<String> names = new ArrayList<String>(bits);
                for (int i = 0; i < bits; ++i) {
                    names.add(name + i);
                }
                this.modelAnalyzerInfo.addOutputBus(s.getName(), names);
            }
            catch (NodeException e) {
                throw new AnalyseException(e);
            }
        }
        return outputs;
    }

    private ArrayList<Signal> checkBinaryInputs(ArrayList<Signal> list) throws AnalyseException {
        ArrayList<Signal> inputs = new ArrayList<Signal>();
        for (final Signal s : list) {
            if (this.ignoreSignal(s)) continue;
            int bits = s.getValue().getBits();
            if (bits == 1) {
                inputs.add(s);
                continue;
            }
            try {
                Splitter sp = Splitter.createNToOne(bits);
                final ObservableValue out = (ObservableValue)sp.getOutputs().get(0);
                out.addObserver(new NodeWithoutDelay(new ObservableValue[]{s.getValue()}){

                    @Override
                    public void hasChanged() {
                        s.getValue().setValue(out.getValue());
                    }
                });
                out.fireHasChanged();
                SplitPinString pins = SplitPinString.create(s);
                ObservableValues.Builder builder = new ObservableValues.Builder();
                String name = s.getName();
                if (!name.contains("_")) {
                    name = name + "_";
                }
                for (int i = bits - 1; i >= 0; --i) {
                    ObservableValue o = new ObservableValue(name + i, 1);
                    builder.add(o);
                    inputs.add(new Signal(name + i, o).setPinNumber(pins.getPin(i)));
                }
                ObservableValues inputsList = builder.reverse().build();
                sp.setInputs(inputsList);
                this.modelAnalyzerInfo.addInputBus(s.getName(), inputsList.getNames());
            }
            catch (NodeException e) {
                throw new AnalyseException(e);
            }
        }
        return inputs;
    }

    private boolean ignoreSignal(Signal s) {
        return s.getName().equals("VCC") || s.getName().equals("GND");
    }

    private void checkClock(Node node) throws AnalyseException {
        if (!this.getClock().hasObserver(node)) {
            throw new AnalyseException(Lang.get("err_ffNeedsToBeConnectedToClock", new Object[0]));
        }
    }

    private ObservableValue getClock() throws AnalyseException {
        ArrayList<Clock> clocks = this.model.getClocks();
        if (clocks.size() != 1) {
            throw new AnalyseException(Lang.get("err_aSingleClockNecessary", new Object[0]));
        }
        return clocks.get(0).getClockOutput();
    }

    private List<FlipflopD> replaceMultiBitFlipflops(List<FlipflopD> flipflops) throws AnalyseException {
        ArrayList<FlipflopD> out = new ArrayList<FlipflopD>();
        for (FlipflopD ff : flipflops) {
            if (ff.getDataBits() == 1) {
                out.add(ff);
                continue;
            }
            try {
                this.model.removeNode(ff);
                ff.getDInput().removeObserver(ff);
                ff.getClock().removeObserver(ff);
                Splitter insp = Splitter.createOneToN(ff.getDataBits());
                insp.setInputs(new ObservableValues(ff.getDInput()));
                ff.getDInput().fireHasChanged();
                Splitter outsp = Splitter.createNToOne(ff.getDataBits());
                ObservableValues.Builder spinput = new ObservableValues.Builder();
                String label = ff.getLabel();
                if (label.length() == 0) {
                    label = this.createOutputBasedName(ff);
                }
                if (!label.contains("_")) {
                    label = label + "_";
                }
                long def = ff.getDefault();
                for (int i = ff.getDataBits() - 1; i >= 0; --i) {
                    ObservableValue qn = new ObservableValue("", 1);
                    ObservableValue nqn = new ObservableValue("", 1);
                    FlipflopD newff = new FlipflopD(label + i, qn, nqn, (def & 1L << i) != 0L ? 1L : 0L);
                    spinput.addAtTop(qn);
                    this.model.add(newff);
                    newff.setInputs(new ObservableValues((ObservableValue)insp.getOutputs().get(i), this.getClock()));
                    out.add(newff);
                }
                outsp.setInputs(spinput.build());
                for (ObservableValue v : spinput) {
                    v.fireHasChanged();
                }
                final ObservableValue qout = (ObservableValue)ff.getOutputs().get(0);
                final ObservableValue nqout = (ObservableValue)ff.getOutputs().get(1);
                final ObservableValue spq = (ObservableValue)outsp.getOutputs().get(0);
                spq.addObserver(new NodeWithoutDelay(new ObservableValue[]{qout, nqout}){

                    @Override
                    public void hasChanged() {
                        long value = spq.getValue();
                        qout.setValue(value);
                        nqout.setValue(value ^ 0xFFFFFFFFFFFFFFFFL);
                    }
                });
                spq.fireHasChanged();
            }
            catch (NodeException e) {
                throw new AnalyseException(e);
            }
        }
        return out;
    }

    public ArrayList<Signal> getInputs() {
        return this.inputs;
    }

    public ArrayList<Signal> getOutputs() {
        return this.outputs;
    }

    public int calcMaxPathLen() throws PinException, BacktrackException, AnalyseException {
        LOGGER.debug("start to calculate the max path len of the model...");
        CycleDetector.checkForCycles(this.inputs);
        PathLenAnalyser da = new PathLenAnalyser(this);
        return da.getMaxPathLen();
    }

    public TruthTable analyse() throws NodeException, PinException, BacktrackException, AnalyseException {
        LOGGER.debug("start to analyse the model...");
        TruthTable tt = new TruthTable();
        tt.setModelAnalyzerInfo(this.getModelAnalyzerInfo());
        for (Signal s : this.inputs) {
            tt.addVariable(s.getName());
        }
        if (!Main.isExperimentalMode() && !this.modelContainsSwitches()) {
            CycleDetector.checkForCycles(this.inputs);
        }
        DependencyAnalyser da = new DependencyAnalyser(this);
        long steps = da.getRequiredSteps(this);
        long tableRows = 1L << this.inputs.size();
        LOGGER.debug("analyse speedup: " + tableRows + " rows vs " + steps + " steps, speedup " + (double)tableRows / (double)steps);
        long time = System.currentTimeMillis();
        if (tableRows <= steps || tableRows <= 128L) {
            this.simpleFiller(tt);
        } else {
            this.dependantFiller(tt, da);
        }
        time = System.currentTimeMillis() - time;
        LOGGER.debug("model analysis: " + (double)time / 1000.0 + " sec");
        return tt;
    }

    private boolean modelContainsSwitches() {
        for (Node n : this.model) {
            if (!(n instanceof Relay) && !(n instanceof RelayDT) && !(n instanceof NFET)) continue;
            return true;
        }
        return false;
    }

    private void simpleFiller(TruthTable tt) throws NodeException, AnalyseException {
        if (this.inputs.size() > 24) {
            throw new AnalyseException(Lang.get("err_toManyInputs_max_N0_is_N1", 24, this.inputs.size()));
        }
        BitSetter bitsetter = new BitSetter(this.inputs.size()){

            @Override
            public void setBit(int row, int bit, boolean value) {
                ((Signal)ModelAnalyser.this.inputs.get(bit)).getValue().setBool(value);
            }
        };
        int rows = 1 << this.inputs.size();
        ArrayList<BoolTableByteArray> data = new ArrayList<BoolTableByteArray>();
        for (Signal s : this.outputs) {
            BoolTableByteArray e = new BoolTableByteArray(rows);
            data.add(e);
            tt.addResult(s.getName(), e);
        }
        this.model.init();
        for (int row = 0; row < rows; ++row) {
            bitsetter.fill(row);
            this.model.doStep();
            for (int i = 0; i < this.outputs.size(); ++i) {
                ((BoolTableByteArray)data.get(i)).set(row, this.outputs.get(i).getValue().getBool());
            }
        }
    }

    private void dependantFiller(TruthTable tt, DependencyAnalyser da) throws NodeException, AnalyseException {
        this.model.init();
        for (Signal out : this.outputs) {
            final ArrayList<Signal> ins = this.reorder(da.getInputs(out), this.inputs);
            if (ins.size() > 24) {
                throw new AnalyseException(Lang.get("err_toManyInputs_max_N0_is_N1", 24, ins.size()));
            }
            int rows = 1 << ins.size();
            BoolTableByteArray e = new BoolTableByteArray(rows);
            BitSetter bitsetter = new BitSetter(ins.size()){

                @Override
                public void setBit(int row, int bit, boolean value) {
                    ((Signal)ins.get(bit)).getValue().setBool(value);
                }
            };
            for (int row = 0; row < rows; ++row) {
                bitsetter.fill(row);
                this.model.doStep();
                e.set(row, out.getValue().getBool());
            }
            tt.addResult(out.getName(), new BoolTableExpanded(e, ins, this.inputs));
        }
    }

    private ModelAnalyserInfo getModelAnalyzerInfo() {
        return this.modelAnalyzerInfo;
    }

    private ArrayList<Signal> reorder(ArrayList<Signal> ins, ArrayList<Signal> originalOrder) {
        ArrayList<Signal> newList = new ArrayList<Signal>();
        for (Signal i : originalOrder) {
            if (!ins.contains(i)) continue;
            newList.add(i);
        }
        return newList;
    }
}

